<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;
use File;

class UserController extends Controller
{
    /**
     * Display the users list.
     */
    public function index(Request $request): Response
    {
        $users = User::where('user_type_id', 3)->paginate(config('app.pagination'));

        return Inertia::render('User/Index', [
            'users' => $users,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the user create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('User/Create');
    }

    /**
     * Display the user edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $user = User::find($id);

        return Inertia::render('User/Update', [
            'user' => $user
        ]);
    }

    /**
     * Delete the user .
     */
    public function delete(Request $request, $id)
    {
        $user = User::find($id);

        if ($user->appointments->count() > 0) {
            return to_route('user.index')->with('error_msg', 'User cannot be deleted, because this user has appointments');
        }

        $user->delete();

        return to_route('user.index')->with('success_msg', 'User has been deleted successfully');
    }

    /**
     * Save new user.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'color_code' => ['required', 'string'],
            'email' => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'profile_image' => ['required', 'image', 'mimes:jpeg,png,jpg', 'max:2048']
        ]);

        $params = Arr::except($request->all(), ['password_confirmation', 'profile_image']);
        $params['password'] = Hash::make($params['password']);
        $params['user_type_id'] = 3;

        if (isset($request->profile_image)) {
            $image_name = time().'.'.$request->profile_image->extension();
            $request->profile_image->move(public_path('images/user-profile-images'), $image_name);
            $params['profile_image'] = '/images/user-profile-images/'.$image_name;
        }

        User::create($params);

        return to_route('user.index')->with('success_msg', 'User has been created successfully');
    }

    /**
     * Update new user.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'color_code' => ['required', 'string'],
            'name' => ['required', 'string', 'max:100'],
            'email' => ['required', 'email', 'unique:users,email,'.$request->id],
            'password' => ['sometimes', 'nullable', 'string', 'min:8', 'confirmed'],
            'profile_image' => ['sometimes', 'image', 'mimes:jpeg,png,jpg', 'max:2048']
        ]);

        $user = User::find($request->id);

        $params = Arr::except($request->all(), ['password_confirmation', 'profile_image']);

        if (isset($request->profile_image)) {
            File::delete(public_path($user->profile_image));
            $image_name = time().'.'.$request->profile_image->extension();
            $request->profile_image->move(public_path('images/user-profile-images'), $image_name);
            $params['profile_image'] = '/images/user-profile-images/'.$image_name;
        }

        if (!empty($request->password)) {
            $params['password'] = Hash::make($request->password);
        }

        $user->update($params);

        return to_route('user.index')->with('success_msg', 'User has been updated successfully');
    }
}
