<?php

namespace App\Http\Controllers;

use App\Models\Sector;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use Inertia\Response;

class SectorController extends Controller
{
    /**
     * Display the sectors list.
     */
    public function index(Request $request): Response
    {
        $sectors = Sector::paginate(config('app.pagination'));

        return Inertia::render('Sector/Index', [
            'sectors' => $sectors,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the sector create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('Sector/Create');
    }

    /**
     * Display the sector edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $sector = Sector::find($id);

        return Inertia::render('Sector/Update', [
            'sector' => $sector
        ]);
    }

    /**
     * Delete the sector .
     */
    public function delete(Request $request, $id)
    {
        $sector = Sector::find($id);

        if ($sector->appointments->count() > 0) {
            return to_route('sector.index')->with('error_msg', 'Sector cannot be deleted, because this sector has appointments');
        }

        $sector->delete();

        return to_route('sector.index')->with('success_msg', 'Sector has been deleted successfully');
    }

    /**
     * Save new sector.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100', 'unique:sectors,name']
        ]);

        Sector::create([
            'name' => $request->name,
        ]);

        return to_route('sector.index')->with('success_msg', 'Sector has been created successfully');
    }

    /**
     * Update new sector.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'name' => ['required', 'string', 'max:100', 'unique:sectors,name,'.$request->id]
        ]);

        $sector = Sector::find($request->id);

        $sector->update([
            'name' => $request->name,
        ]);

        return to_route('sector.index')->with('success_msg', 'Sector has been updated successfully');
    }
}
