<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use Inertia\Response;

class RoleController extends Controller
{
    /**
     * Display the roles list.
     */
    public function index(Request $request): Response
    {
        $roles = Role::paginate(config('app.pagination'));

        return Inertia::render('Role/Index', [
            'roles' => $roles,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the role create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('Role/Create');
    }

    /**
     * Display the role edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $role = Role::find($id);

        return Inertia::render('Role/Update', [
            'role' => $role
        ]);
    }

    /**
     * Delete the role .
     */
    public function delete(Request $request, $id)
    {
        $role = Role::find($id);

        if ($role->users->count() > 0) {
            return to_route('role.index')->with('error_msg', 'Role cannot be deleted, because this role is assigned to users');
        }

        $role->permissions()->sync([]);
        $role->delete();

        return to_route('role.index')->with('success_msg', 'Role has been deleted successfully');
    }

    /**
     * Save new role.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:100', 'unique:roles,title']
        ]);

        $name = strtolower( str_replace(" ", "_", $request->title) );

        Role::create([
            'title' => $request->title,
            'name' => $name,
        ]);

        return to_route('role.index')->with('success_msg', 'Role has been created successfully');
    }

    /**
     * Update new role.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'title' => ['required', 'string', 'max:100', 'unique:roles,title,'.$request->id]
        ]);

        $role = Role::find($request->id);

        $name = strtolower( str_replace(" ", "_", $request->title) );

        $role->update([
            'title' => $request->title,
            'name' => $name,
        ]);

        return to_route('role.index')->with('success_msg', 'Role has been updated successfully');
    }
}
