<?php

namespace App\Http\Controllers;

use App\Models\BlockedPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use Inertia\Response;

class BlockedPeriodController extends Controller
{
    /**
     * Display the blocked periods list.
     */
    public function index(Request $request): Response
    {
        $blocked_periods = BlockedPeriod::paginate(config('app.pagination'));

        return Inertia::render('BlockedPeriod/Index', [
            'blocked_periods' => $blocked_periods,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the blocked period create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('BlockedPeriod/Create');
    }

    /**
     * Display the blocked period edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $blocked_period = BlockedPeriod::find($id);

        return Inertia::render('BlockedPeriod/Update', [
            'blocked_period' => $blocked_period
        ]);
    }

    /**
     * Delete the blocked period.
     */
    public function delete(Request $request, $id)
    {
        $blocked_period = BlockedPeriod::find($id);

        $blocked_period->delete();

        return to_route('blocked_period.index')->with('success_msg', 'Blocked period has been deleted successfully');
    }

    /**
     * Save new blocked period.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:200'],
            'start_date' => ['required', 'date'],
            'end_date' => ['required', 'date', 'after_or_equal:start_date'],
        ]);

        BlockedPeriod::create([
            'title' => $request->title,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ]);

        return to_route('blocked_period.index')->with('success_msg', 'Blocked period has been created successfully');
    }

    /**
     * Update new blocked period.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'title' => ['required', 'string', 'max:200'],
            'start_date' => ['required', 'date'],
            'end_date' => ['required', 'date', 'after_or_equal:start_date'],
        ]);

        $blocked_period = BlockedPeriod::find($request->id);

        $blocked_period->update([
            'title' => $request->title,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ]);

        return to_route('blocked_period.index')->with('success_msg', 'Blocked period has been updated successfully');
    }
}
