<?php

namespace App\Http\Controllers;

use App\Exports\AppointmentExport;
use App\Models\Appointment;
use App\Models\AppointmentTimeline;
use App\Models\Sector;
use App\Rules\IsDateInBlockedPeriod;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Inertia\Inertia;
use Inertia\Response;

class AppointmentController extends Controller
{
    /**
     * Display the appointments list.
     */
    public function index(Request $request): Response
    {
        $appointments = Appointment::with('sector', 'createdBy');

        if (!empty($request->organization)) {
            $appointments->where('organization', $request->organization);
        }

        if (!empty($request->sector_id)) {
            $appointments->where('sector_id', $request->sector_id);
        }

        if (!empty($request->volunteer)) {
            $appointments->where('volunteer', 'LIKE', '%'.$request->volunteer.'%');
        }

        if (!empty($request->professional)) {
            $appointments->where('professional', $request->professional);
        }

        if (!empty($request->client)) {
            $appointments->where('client', $request->client);
        }

        if (!empty($request->appointment_date)) {
            $appointments->whereDate('appointment_date', $request->appointment_date);
        }

        if (!empty($request->appointment_date_from)) {
            $appointments->whereDate('appointment_date', '>=', $request->appointment_date_from);
        }

        if (!empty($request->appointment_date_to)) {
            $appointments->whereDate('appointment_date', '<=', $request->appointment_date_to);
        }
        
        $appointments = $appointments->orderBy('appointment_date', 'desc')
        ->paginate(config('app.pagination'));

        $sectors = Sector::get();

        return Inertia::render('Appointment/Index', [
            'appointments' => $appointments,
            'sectors' => $sectors,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the appointment create form.
     */
    public function create(Request $request): Response
    {
        $sectors = Sector::get();

        return Inertia::render('Appointment/Create',[
            'sectors' => $sectors,
        ]);
    }

    /**
     * Display the appointment edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $appointment = Appointment::find($id);

        $sectors = Sector::get();

        return Inertia::render('Appointment/Update', [
            'appointment' => $appointment,
            'sectors' => $sectors
        ]);
    }

    /**
     * Display the appointment detail.
     */
    public function detail(Request $request, $id): Response
    {
        $appointment = Appointment::with([
            'sector', 
            'appointmentTimelines.actionBy'
        ])->find($id);

        $appointment->appointment_date = Carbon::parse($appointment->appointment_date)->format('d-m-Y');

        return Inertia::render('Appointment/Detail', [
            'appointment' => $appointment
        ]);
    }

    /**
     * Delete the appointment .
     */
    public function delete(Request $request, $id)
    {
        $appointment = Appointment::find($id);

        return DB::transaction(function () use ($appointment) {
            $appointment->appointmentTimelines()->delete();
            $appointment->delete();

            return to_route('appointment.index')->with('success_msg', 'Appointment has been deleted successfully');
        });
    }

    /**
     * Save new appointment.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:200'],
            'professional_person_email' => ['required', 'email'],
            'description' => ['required', 'string', 'max:500'],
            'sector_id' => ['required', 'numeric', 'exists:sectors,id'],
            'organization' => ['required', 'string', 'max:200'],
            'volunteer' => ['required', 'string', 'max:200'],
            'professional' => ['required', 'string', 'max:200'],
            'client' => ['required', 'string', 'max:200'],
            'no_of_meeting' => ['required', 'numeric', 'min:1'],
            'appointment_date' => ['required', 'date', new IsDateInBlockedPeriod()],
            'appointment_start_time' => ['required', 'date_format:H:i'],
            'appointment_end_time' => ['required', 'date_format:H:i', 'after:appointment_start_time']
        ]);

        $appointment_params = $request->all();
        $appointment_params['appointment_number'] = $this->getAppointmentNumber();
        $appointment_params['created_by'] = Auth::user()->id;

        return DB::transaction(function () use ($appointment_params) {
            $appointment = Appointment::create($appointment_params);

            AppointmentTimeline::create([
                'appointment_id' => $appointment->id,
                'description' => 'New appointment '.$appointment->appointment_number.' has been created',
                'action_by' => Auth::user()->id,
            ]);

            return to_route('appointment.index')->with('success_msg', 'Appointment has been created successfully');
        });
    }

    /**
     * Update new appointment.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'title' => ['required', 'string', 'max:200'],
            'professional_person_email' => ['required', 'email'],
            'description' => ['required', 'string', 'max:500'],
            'sector_id' => ['required', 'numeric', 'exists:sectors,id'],
            'organization' => ['required', 'string', 'max:200'],
            'volunteer' => ['required', 'string', 'max:200'],
            'professional' => ['required', 'string', 'max:200'],
            'client' => ['required', 'string', 'max:200'],
            'no_of_meeting' => ['required', 'numeric', 'min:1'],
            'appointment_date' => ['required', 'date', new IsDateInBlockedPeriod()],
            'appointment_start_time' => ['required', 'date_format:H:i'],
            'appointment_end_time' => ['required', 'date_format:H:i', 'after:appointment_start_time']
        ]);

        $appointment = Appointment::find($request->id);
        $appointment_params = $request->all();

        return DB::transaction(function () use ($appointment, $appointment_params) {
            $appointment->update($appointment_params);

            AppointmentTimeline::create([
                'appointment_id' => $appointment->id,
                'description' => 'Appointment '.$appointment->appointment_number.' has been updated',
                'action_by' => Auth::user()->id,
            ]);

            return to_route('appointment.index')->with('success_msg', 'Appointment has been updated successfully');
        });
    }

    /**
     * Update appointment status.
     */
    public function updateStatus(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'status' => ['required', 'string', 'in:Pending,Approved,Decline,Done'],
        ]);

        $appointment = Appointment::find($request->id);

        return DB::transaction(function () use ($appointment, $request) {
            $appointment->update(['status' => $request->status]);

            AppointmentTimeline::create([
                'appointment_id' => $appointment->id,
                'description' => 'Appointment '.$appointment->appointment_number.' status has been updated to '.$request->status,
                'action_by' => Auth::user()->id,
            ]);

            return to_route('appointment.index')->with('success_msg', 'Appointment status has been updated successfully');
        });
    }

    /**
     * Export appointment in excel file.
     */
    public function export(Request $request)
    {
        $filters = $request->all();
        $appointments = new AppointmentExport($filters);
        return ($appointments)->download('appointments.xlsx');
    }

    private function getAppointmentNumber()
    {
        $appointmentCount = Appointment::count();
        $appointmentCount++;
        return 'appointment-'.str_pad($appointmentCount, 6, '0', STR_PAD_LEFT);
    }
}
